// Background Service Worker - MV3 Compatible
// Handles all API communication (CSP-safe)

// Default API URL (production)
const DEFAULT_API_BASE_URL = 'https://autofill.altiq.io/api/v1';

// Current API URL (loaded from storage)
let API_BASE_URL = DEFAULT_API_BASE_URL;

let DEBUG_LOGGING = false;

function logDebug(...args) {
  if (DEBUG_LOGGING) console.log(...args);
}

logDebug('Altiq AutoFill background service worker started');

function normalizeApiBaseUrl(raw) {
  const url = new URL(String(raw || '').trim());
  if (!['http:', 'https:'].includes(url.protocol)) throw new Error('Invalid protocol');
  const path = url.pathname.replace(/\/+$/, '');
  if (!path.endsWith('/api/v1')) throw new Error('Invalid path');
  url.pathname = path;
  url.hash = '';
  return url.toString();
}

function isAllowedApiBaseUrl(normalizedUrl) {
  const url = new URL(normalizedUrl);
  const allow = new Set([
    'https://autofill.altiq.io/api/v1',
    'https://dev-autofill.altiq.io/api/v1',
    'http://localhost:8001/api/v1',
    'http://127.0.0.1:8001/api/v1',
    'http://[::1]:8001/api/v1'
  ]);
  const isLoopbackHost = ['localhost', '127.0.0.1', '::1', '[::1]'].includes(url.hostname);
  if (isLoopbackHost && url.protocol === 'http:' && url.pathname.endsWith('/api/v1')) return true;
  return allow.has(url.toString());
}

// Load API configuration from storage
async function loadApiConfig() {
  const result = await chrome.storage.local.get(['api_server_type', 'api_base_url', 'debug_logging']);
  const serverType = result.api_server_type || 'production';

  DEBUG_LOGGING = Boolean(result.debug_logging);
  
  if (serverType === 'production') {
    API_BASE_URL = DEFAULT_API_BASE_URL;
  } else {
    try {
      const normalized = normalizeApiBaseUrl(result.api_base_url || DEFAULT_API_BASE_URL);
      API_BASE_URL = isAllowedApiBaseUrl(normalized) ? normalized : DEFAULT_API_BASE_URL;
    } catch {
      API_BASE_URL = DEFAULT_API_BASE_URL;
    }
  }

  logDebug('[BG] API configured:', API_BASE_URL);
  return API_BASE_URL;
}

// Initialize config on startup
loadApiConfig();

// Get stored auth token
async function getToken() {
  const result = await chrome.storage.local.get('token');
  return result.token || null;
}

async function getAuthState() {
  const result = await chrome.storage.local.get(['auth_mode', 'token', 'aura_session_id', 'aura_tenant_id']);
  const mode = result.auth_mode || (result.token ? 'jwt' : (result.aura_session_id ? 'aura' : null));
  return {
    mode,
    token: result.token || null,
    auraSessionId: result.aura_session_id || null,
    auraTenantId: result.aura_tenant_id || null
  };
}

function formatErrorMessage(err) {
  if (!err) return 'Unknown error';
  if (typeof err === 'string') return err;
  if (err instanceof Error) return err.message || 'Unknown error';

  const detail = err.detail ?? err.message ?? err.error;
  if (typeof detail === 'string') return detail;
  if (Array.isArray(detail)) {
    const msgs = detail
      .map(d => (typeof d === 'string' ? d : (d?.msg || d?.message || JSON.stringify(d))))
      .filter(Boolean);
    return msgs.length ? msgs.join(' | ') : 'Request failed';
  }
  if (detail && typeof detail === 'object') return JSON.stringify(detail);
  try {
    return JSON.stringify(err);
  } catch {
    return String(err);
  }
}

// Make authenticated API request
async function apiRequest(endpoint, options = {}) {
  // Ensure config is loaded
  await loadApiConfig();

  const auth = await getAuthState();
  logDebug('[BG] API Request:', endpoint, 'Auth mode:', auth.mode);

  if (!auth.mode) {
    throw new Error('Not authenticated. Please login first.');
  }
  
  const url = `${API_BASE_URL}${endpoint}`;
  const headers = {
    'Content-Type': 'application/json',
    ...options.headers
  };

  if (auth.mode === 'jwt') {
    if (!auth.token) throw new Error('Not authenticated. Please login first.');
    headers['Authorization'] = `Bearer ${auth.token}`;
  } else if (auth.mode === 'aura') {
    if (!auth.auraSessionId) throw new Error('Not authenticated. Please login first.');
    headers['X-Session-Id'] = auth.auraSessionId;
    if (auth.auraTenantId) headers['X-Tenant-Id'] = auth.auraTenantId;
  }
  
  logDebug('[BG] Fetching:', url, 'Method:', options.method || 'GET');
  
  try {
    const response = await fetch(url, {
      ...options,
      headers
    });
    
    logDebug('[BG] Response status:', response.status);
    
    if (!response.ok) {
      if (response.status === 401) {
        // Token expired
        await chrome.storage.local.remove(['token', 'user', 'auth_mode', 'aura_session_id', 'aura_tenant_id']);
        throw new Error('Session expired. Please login again.');
      }
      
      const error = await response.json().catch(() => ({ detail: response.statusText }));
      throw new Error(formatErrorMessage(error) || `API error: ${response.status}`);
    }
    
    return await response.json();
  } catch (error) {
    console.error('[BG] Fetch error:', error);
    throw error;
  }
}

// Message handlers
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  logDebug('[BG] Background received message:', message?.type || message?.action);
  
  // Handle config reload from options page
  if (message.type === 'RELOAD_CONFIG') {
    loadApiConfig().then(() => {
      logDebug('[BG] Config reloaded:', API_BASE_URL);
      sendResponse({ success: true, url: API_BASE_URL });
    });
    return true;
  }
  
  // Handle different message types
  if (message.type === 'ALTIQ_GET_CONTEXT') {
    logDebug('[BG] Handling GET_CONTEXT');
    handleGetContext()
      .then(result => {
        logDebug('[BG] Context result received');
        sendResponse(result);
      })
      .catch(error => {
        console.error('[BG] Context error:', error);
        sendResponse({ error: error.message });
      });
    return true;
  }
  
  if (message.type === 'ALTIQ_MAP_FIELDS') {
    logDebug('[BG] Handling MAP_FIELDS, field count:', message.payload?.fields?.length);
    handleMapFields(message.payload)
      .then(result => {
        logDebug('[BG] Map fields result received');
        sendResponse(result);
      })
      .catch(error => {
        console.error('[BG] Map fields error:', error);
        sendResponse({ error: error.message });
      });
    return true;
  }
  
  if (message.type === 'ALTIQ_SET_ACTIVE_DOC') {
    handleSetActiveDoc(message.payload)
      .then(result => sendResponse(result))
      .catch(error => sendResponse({ error: error.message }));
    return true;
  }
  
  if (message.type === 'ALTIQ_LEARN_MAPPING') {
    handleLearnMapping(message.payload)
      .then(result => sendResponse(result))
      .catch(error => sendResponse({ error: error.message }));
    return true;
  }
  
  // Legacy support
  if (message.action === 'getToken') {
    getToken().then(token => sendResponse({ token }));
    return true;
  }
  
  if (message.action === 'apiRequest') {
    apiRequest(message.config.endpoint, message.config)
      .then(data => sendResponse({ success: true, data }))
      .catch(error => sendResponse({ success: false, error: error.message }));
    return true;
  }
  
  return false;
});

/**
 * Get autofill context (documents, active doc, profile)
 */
async function handleGetContext() {
  try {
    const [documents, fields] = await Promise.all([
      apiRequest('/documents/'),
      apiRequest('/fields/')
    ]);
    
    const hasDocuments = documents && documents.length > 0;
    const latestDoc = hasDocuments ? documents[documents.length - 1] : null;
    
    // Get active doc from storage or use latest
    const storage = await chrome.storage.local.get('active_doc_id');
    const activeDocId = storage.active_doc_id || latestDoc?.id || null;
    
    // Build profile summary (safe fields only)
    const profile = {};
    if (fields && fields.length > 0) {
      fields.forEach(field => {
        // Skip sensitive fields
        if (field.field_type === 'ssn' || field.field_name.toLowerCase().includes('password')) {
          return;
        }
        profile[field.field_name] = field.field_value;
      });
    }
    
    return {
      has_documents: hasDocuments,
      latest_doc_id: latestDoc?.id || null,
      active_doc_id: activeDocId,
      document_count: documents.length,
      field_count: fields.length,
      profile_summary: profile
    };
  } catch (error) {
    console.error('Failed to get context:', error);
    throw error;
  }
}

/**
 * Map detected fields to user data and return fill plan
 */
async function handleMapFields(payload) {
  try {
    const { url, hostname, title, fields } = payload;

    logDebug(`Mapping ${fields.length} fields for ${hostname}`);
    
    // STEP 1: Get context for this hostname (to get available clients and scope)
    let selectedScope = 'both';
    let selectedClientId = null;
    
    try {
      logDebug(`[CONTEXT] Fetching context for hostname: ${hostname}`);
      const contextResponse = await apiRequest(`/autofill/context?hostname=${encodeURIComponent(hostname)}`, {
        method: 'GET'
      });

      logDebug(`[CONTEXT] Retrieved context for ${hostname}`);
      
      selectedScope = contextResponse.selected_scope || 'both';
      selectedClientId = contextResponse.selected_client_id;
      
      logDebug(`[CONTEXT] Scope from context: ${selectedScope}, Client from context: ${selectedClientId}`);
      
      // If no client was previously selected but clients are available, use the first one
      if (!selectedClientId && contextResponse.clients && contextResponse.clients.length > 0) {
        selectedClientId = contextResponse.clients[0].id;
        logDebug(`[CONTEXT] Auto-selected first client: ${selectedClientId} (from ${contextResponse.clients.length} available)`);
        
        // If scope is 'enterprise' but we have clients and no enterprise profile, switch to 'client'
        if (selectedScope === 'enterprise' && !contextResponse.has_enterprise_profile && contextResponse.clients.length > 0) {
          selectedScope = 'client';
          logDebug(`[CONTEXT] Changed scope from 'enterprise' to 'client' (no enterprise profile, has client profiles)`);
        }
      } else if (contextResponse.clients && contextResponse.clients.length === 0) {
        logDebug(`[CONTEXT] No clients available for user. User needs to create a client profile first.`);
      }
    } catch (contextError) {
      console.error('[CONTEXT] Failed to get context:', contextError);
      logDebug(`[CONTEXT] Falling back to defaults: scope='both', no client`);
      // Continue with defaults (both scope, no client)
    }
    
    // STEP 2: Call backend mapping endpoint with context
    const response = await apiRequest('/autofill/map-fields', {
      method: 'POST',
      body: JSON.stringify({
        url,
        hostname,
        title,
        fields,
        selected_scope: selectedScope,
        selected_client_id: selectedClientId
      })
    });
    
    logDebug('[MAP-FIELDS] Response received');
    
    return {
      fillPlan: response.fill_plan,
      summary: response.summary
    };
  } catch (error) {
    console.error('Field mapping failed:', error);
    throw error;
  }
}

/**
 * Set active document for autofill
 */
async function handleSetActiveDoc(payload) {
  try {
    const { document_id } = payload;
    
    // Store locally
    await chrome.storage.local.set({ active_doc_id: document_id });
    
    // Optionally call backend to store preference
    await apiRequest('/autofill/active-doc', {
      method: 'POST',
      body: JSON.stringify({ document_id })
    });
    
    return { success: true, document_id };
  } catch (error) {
    console.error('Failed to set active doc:', error);
    throw error;
  }
}

/**
 * Learn from user corrections
 */
async function handleLearnMapping(payload) {
  try {
    const { url, hostname, field_signature, corrected_value, canonical_key } = payload;
    
    // Send learning data to backend
    await apiRequest('/autofill/learn', {
      method: 'POST',
      body: JSON.stringify({
        url,
        hostname,
        field_signature,
        corrected_value,
        canonical_key
      })
    });
    
    return { success: true };
  } catch (error) {
    console.error('Learning failed:', error);
    throw error;
  }
}

// Context menu for quick autofill - only create on install/update
chrome.runtime.onInstalled.addListener(() => {
  chrome.contextMenus.create({
    id: 'autofill-page',
    title: 'AutoFill this page',
    contexts: ['page']
  });
});

chrome.contextMenus.onClicked.addListener(async (info, tab) => {
  if (info.menuItemId === 'autofill-page') {
    const auth = await getAuthState();
    
    if (!auth.mode) {
      chrome.action.openPopup();
      return;
    }
    
    // Inject on demand for least privilege (no always-on content script).
    chrome.scripting.insertCSS({ target: { tabId: tab.id }, files: ['content.css'] }, () => {
      chrome.scripting.executeScript({ target: { tabId: tab.id }, files: ['content.js'] }, () => {
        chrome.tabs.sendMessage(tab.id, { action: 'autofill' });
      });
    });
  }
});

logDebug('Background service worker ready');
