// Content Script - CSP-Safe Form Field Detection and AutoFill
// NO DIRECT FETCH CALLS - All API requests go through background.js
let DEBUG_LOGGING = false;

function logDebug(...args) {
  if (DEBUG_LOGGING) console.log(...args);
}

chrome.storage?.local?.get?.(['debug_logging'], (result) => {
  DEBUG_LOGGING = Boolean(result?.debug_logging);
  logDebug('Altiq AutoFill content script loaded');
});

// Confidence level colors
const CONFIDENCE_COLORS = {
  high: '#48bb78',    // Green (>= 0.8)
  medium: '#f6ad55',  // Orange (>= 0.5)
  low: '#fc8181'      // Red (< 0.5)
};

/**
 * Detect all form fields and extract comprehensive metadata
 */
function detectFields() {
  const detectedFields = [];
  const inputs = document.querySelectorAll('input, textarea, select');
  
  inputs.forEach((input, index) => {
    // Skip hidden fields, buttons, and submit inputs
    const type = input.type?.toLowerCase();
    if (type === 'hidden' || type === 'submit' || type === 'button' || type === 'image') {
      return;
    }
    
    const fieldInfo = analyzeField(input, index);
    if (fieldInfo) {
      detectedFields.push(fieldInfo);
    }
  });
  
  logDebug(`Detected ${detectedFields.length} fillable fields`);
  return detectedFields;
}

/**
 * Analyze a single field and extract all metadata for mapping
 */
function analyzeField(element, index) {
  const id = element.id || '';
  const name = element.name || '';
  const placeholder = element.placeholder || '';
  const type = element.type || 'text';
  const tagName = element.tagName.toLowerCase();
  const autocomplete = element.autocomplete || '';
  const ariaLabel = element.getAttribute('aria-label') || '';
  const required = element.required || false;
  
  // Get associated label
  let labelText = '';
  if (element.id) {
    const labelEl = document.querySelector(`label[for="${element.id}"]`);
    if (labelEl) {
      labelText = labelEl.textContent.trim();
    }
  }
  
  // If no label found, look for nearby label or text
  if (!labelText) {
    const parent = element.parentElement;
    if (parent) {
      const parentLabel = parent.querySelector('label');
      if (parentLabel) {
        labelText = parentLabel.textContent.trim();
      } else {
        // Get nearby text (within parent)
        const clone = parent.cloneNode(true);
        // Remove child inputs to avoid their values
        clone.querySelectorAll('input, select, textarea').forEach(el => el.remove());
        labelText = clone.textContent.trim();
        if (labelText.length > 100) {
          labelText = labelText.substring(0, 100) + '...';
        }
      }
    }
  }
  
  // Get nearby text (siblings)
  let nearbyText = '';
  const prevSibling = element.previousElementSibling;
  const nextSibling = element.nextElementSibling;
  if (prevSibling && prevSibling.tagName !== 'INPUT' && prevSibling.tagName !== 'SELECT') {
    nearbyText = prevSibling.textContent.trim();
  } else if (nextSibling && nextSibling.tagName !== 'INPUT' && nextSibling.tagName !== 'SELECT') {
    nearbyText = nextSibling.textContent.trim();
  }
  if (nearbyText.length > 50) {
    nearbyText = nearbyText.substring(0, 50);
  }
  
  // Create a unique selector for this field
  const selector = createSelector(element, index);
  
  // Create field signature for backend mapping
  const fieldSignature = {
    selector,
    tag: tagName,
    type,
    name,
    id,
    placeholder,
    autocomplete,
    ariaLabel,
    labelText,
    nearbyText,
    required
  };
  
  return fieldSignature;
}

/**
 * Create a robust CSS selector for an element
 */
function createSelector(element, index) {
  // Prefer ID
  if (element.id) {
    return `#${element.id}`;
  }
  
  // Try name attribute
  if (element.name) {
    const tag = element.tagName.toLowerCase();
    const type = element.type ? `[type="${element.type}"]` : '';
    return `${tag}[name="${element.name}"]${type}`;
  }
  
  // Fallback to nth-of-type with tag
  const tag = element.tagName.toLowerCase();
  const parent = element.parentElement;
  if (parent) {
    const siblings = Array.from(parent.children).filter(el => el.tagName === element.tagName);
    const nthIndex = siblings.indexOf(element) + 1;
    return `${tag}:nth-of-type(${nthIndex})`;
  }
  
  // Last resort: use data attribute we add
  const fallbackId = `altiq-field-${index}`;
  element.setAttribute('data-altiq-id', fallbackId);
  return `[data-altiq-id="${fallbackId}"]`;
}

/**
 * Apply fill plan received from background/backend
 */
function applyFillPlan(fillPlan) {
  if (!fillPlan || !Array.isArray(fillPlan)) {
    console.error('Invalid fill plan received');
    return { filled: 0, skipped: 0, errors: 0 };
  }
  
  let filled = 0;
  let skipped = 0;
  let errors = 0;
  
  fillPlan.forEach(plan => {
    if (plan.actionType === 'skip') {
      skipped++;
      logDebug(`Skipping field ${plan.selector}: ${plan.reason}`);
      return;
    }
    
    // Support both 'fill' (old) and 'setValue' (new) action types
    if (plan.actionType === 'fill' || plan.actionType === 'setValue' || plan.actionType === 'selectOption') {
      try {
        const element = document.querySelector(plan.selector);
        if (!element) {
          logDebug(`Element not found for selector: ${plan.selector}`);
          errors++;
          return;
        }
        
        // Use fill_value (new) or value (old)
        const fillValue = plan.fill_value || plan.value;
        fillField(element, fillValue, plan.confidence);
        filled++;
        logDebug(`Filled ${plan.selector} (${plan.canonicalKey}): confidence ${plan.confidence}`);
      } catch (error) {
        console.error(`Error filling field ${plan.selector}:`, error);
        errors++;
      }
    }
  });
  
  return { filled, skipped, errors };
}

/**
 * Fill a single field with value
 */
function fillField(element, value, confidence) {
  const tagName = element.tagName.toLowerCase();
  
  if (tagName === 'select') {
    // For select elements, try to match option
    const options = Array.from(element.options);
    const match = options.find(opt => 
      opt.value.toLowerCase() === value.toLowerCase() ||
      opt.text.toLowerCase() === value.toLowerCase()
    );
    if (match) {
      element.value = match.value;
    }
  } else {
    element.value = value;
  }
  
  // Trigger events for frameworks (React, Vue, etc.)
  element.dispatchEvent(new Event('input', { bubbles: true }));
  element.dispatchEvent(new Event('change', { bubbles: true }));
  element.dispatchEvent(new Event('blur', { bubbles: true }));
  
  // Visual feedback
  highlightField(element, confidence);
}

/**
 * Highlight a field with confidence color
 */
function highlightField(element, confidence) {
  const color = getConfidenceColor(confidence);
  element.style.borderColor = color;
  element.style.borderWidth = '2px';
  element.style.borderStyle = 'solid';
  element.style.transition = 'all 0.3s ease';
  element.style.boxShadow = `0 0 5px ${color}`;
  
  // Fade out after 3 seconds
  setTimeout(() => {
    element.style.borderColor = '';
    element.style.borderWidth = '';
    element.style.borderStyle = '';
    element.style.boxShadow = '';
  }, 3000);
}

/**
 * Get color based on confidence level
 */
function getConfidenceColor(confidence) {
  if (confidence >= 0.8) return CONFIDENCE_COLORS.high;
  if (confidence >= 0.5) return CONFIDENCE_COLORS.medium;
  return CONFIDENCE_COLORS.low;
}

/**
 * Show notification to user
 */
function showNotification(message, type = 'info') {
  const notification = document.createElement('div');
  notification.className = 'altiq-notification';
  
  const colors = {
    success: '#48bb78',
    warning: '#f6ad55',
    error: '#fc8181',
    info: '#667eea'
  };
  
  notification.style.cssText = `
    position: fixed;
    top: 20px;
    right: 20px;
    background: ${colors[type]};
    color: white;
    padding: 16px 24px;
    border-radius: 8px;
    box-shadow: 0 4px 12px rgba(0,0,0,0.15);
    z-index: 10001;
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Oxygen, Ubuntu, sans-serif;
    font-size: 14px;
    font-weight: 500;
    animation: slideIn 0.3s ease;
    max-width: 400px;
  `;
  
  notification.textContent = message;
  document.body.appendChild(notification);
  
  setTimeout(() => {
    notification.style.animation = 'slideOut 0.3s ease';
    setTimeout(() => notification.remove(), 300);
  }, 4000);
}

/**
 * Main autofill function - coordinates with background script
 */
async function performAutofill(scope, clientId, hostname) {
  logDebug('[CONTENT] Starting autofill process...', { scope, clientId, hostname });
  
  try {
    // Step 1: Detect all fields on the page
    const fields = detectFields();
    logDebug('[CONTENT] Detected fields:', fields.length);
    
    if (fields.length === 0) {
      showNotification('No fillable fields found on this page', 'info');
      return { success: false, message: 'No fields detected' };
    }
    
    showNotification(`Detected ${fields.length} fields, analyzing...`, 'info');
    
    // Step 2: Send field data to background for mapping
    logDebug('[CONTENT] Sending MAP_FIELDS message to background');
    const response = await chrome.runtime.sendMessage({
      type: 'ALTIQ_MAP_FIELDS',
      payload: {
        url: window.location.href,
        hostname: hostname || window.location.hostname,
        title: document.title,
        fields: fields,
        selected_scope: scope,  // Pass scope from popup
        selected_client_id: clientId  // Pass client ID from popup
      }
    });
    
    logDebug('[CONTENT] Received response from background');
    
    if (!response || response.error) {
      const errorMsg = response?.error || 'Failed to map fields';
      console.error('[CONTENT] Mapping error:', errorMsg);
      showNotification(errorMsg, 'error');
      return { success: false, message: errorMsg };
    }
    
    // Step 3: Apply the fill plan
    const fillPlan = response.fillPlan;
    const stats = applyFillPlan(fillPlan);
    
    // Step 4: Show results
    const message = `✓ Filled ${stats.filled} fields, Skipped ${stats.skipped}${stats.errors > 0 ? `, Errors ${stats.errors}` : ''}`;
    showNotification(message, stats.filled > 0 ? 'success' : 'warning');
    
    return { 
      success: true, 
      ...stats,
      total: fields.length
    };
    
  } catch (error) {
    console.error('Autofill error:', error);
    showNotification('Autofill failed: ' + error.message, 'error');
    return { success: false, message: error.message };
  }
}

// Add animation styles
const style = document.createElement('style');
style.textContent = `
  @keyframes slideIn {
    from {
      transform: translateX(400px);
      opacity: 0;
    }
    to {
      transform: translateX(0);
      opacity: 1;
    }
  }
  
  @keyframes slideOut {
    from {
      transform: translateX(0);
      opacity: 1;
    }
    to {
      transform: translateX(400px);
      opacity: 0;
    }
  }
`;
document.head.appendChild(style);

// Listen for messages from popup and background
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  logDebug('Content script received message:', message?.action);
  
  if (message.action === 'autofill') {
    // Pass scope and clientId from popup message
    const scope = message.scope || 'enterprise';
    const clientId = message.clientId || null;
    const hostname = message.hostname || null;
    
    performAutofill(scope, clientId, hostname).then(result => {
      sendResponse(result);
    }).catch(error => {
      sendResponse({ success: false, message: error.message });
    });
    return true; // Keep channel open for async response
  }
  
  if (message.action === 'detectFields') {
    const fields = detectFields();
    sendResponse({ success: true, fields, count: fields.length });
    return false;
  }
  
  return false;
});

logDebug('Altiq AutoFill content script ready (CSP-safe mode)');
