/**
 * Altiq AutoFill - Extension Popup Script
 * Handles authentication, profile management, and autofill actions
 */

// ==================== Configuration ====================
const DEFAULT_API_BASE_URL = 'https://autofill.altiq.io/api/v1';
const DEFAULT_AURA_BASE_URL = 'https://api.altiq.io';

// Best-guess dev Aura base (can be overridden via chrome.storage.local aura_base_url)
const DEFAULT_DEV_AURA_BASE_URL = 'https://dev-admin.altiq.io';

let API_BASE_URL = DEFAULT_API_BASE_URL;
let AURA_BASE_URL = DEFAULT_AURA_BASE_URL;

let DEBUG_LOGGING = false;

function logDebug(...args) {
  if (DEBUG_LOGGING) console.log(...args);
}

function normalizeApiBaseUrl(raw) {
  const url = new URL(String(raw || '').trim());
  if (!['http:', 'https:'].includes(url.protocol)) throw new Error('Invalid protocol');
  const path = url.pathname.replace(/\/+$/, '');
  if (!path.endsWith('/api/v1')) throw new Error('Invalid path');
  url.pathname = path;
  url.hash = '';
  return url.toString();
}

function isAllowedApiBaseUrl(normalizedUrl) {
  const url = new URL(normalizedUrl);
  const allow = new Set([
    'https://autofill.altiq.io/api/v1',
    'https://dev-autofill.altiq.io/api/v1',
    'http://localhost:8001/api/v1',
    'http://127.0.0.1:8001/api/v1',
    'http://[::1]:8001/api/v1'
  ]);
  const isLoopbackHost = ['localhost', '127.0.0.1', '::1', '[::1]'].includes(url.hostname);
  if (isLoopbackHost && url.protocol === 'http:' && url.pathname.endsWith('/api/v1')) return true;
  return allow.has(url.toString());
}

function normalizeAuraBaseUrl(raw) {
  const url = new URL(String(raw || '').trim());
  if (url.protocol !== 'https:') throw new Error('Aura base must be https');
  url.pathname = '';
  url.search = '';
  url.hash = '';
  return url.toString().replace(/\/+$/, '');
}

function isAllowedAuraBaseUrl(normalizedUrl) {
  const allow = new Set([
    'https://api.altiq.io',
    'https://admin.altiq.io',
    'https://dev-admin.altiq.io'
  ]);
  return allow.has(normalizedUrl);
}

async function loadApiConfig() {
  const result = await chrome.storage.local.get(['api_server_type', 'api_base_url', 'aura_base_url', 'debug_logging']);
  const serverType = result.api_server_type || 'production';

  DEBUG_LOGGING = Boolean(result.debug_logging);

  if (serverType === 'production') {
    API_BASE_URL = DEFAULT_API_BASE_URL;
  } else {
    // Strictly allowlist configurable endpoints to prevent accidental/exfil endpoints.
    try {
      const normalized = normalizeApiBaseUrl(result.api_base_url || DEFAULT_API_BASE_URL);
      API_BASE_URL = isAllowedApiBaseUrl(normalized) ? normalized : DEFAULT_API_BASE_URL;
    } catch {
      API_BASE_URL = DEFAULT_API_BASE_URL;
    }
  }

  if (result.aura_base_url) {
    try {
      const normalizedAura = normalizeAuraBaseUrl(result.aura_base_url);
      AURA_BASE_URL = isAllowedAuraBaseUrl(normalizedAura)
        ? normalizedAura
        : (serverType === 'production' ? DEFAULT_AURA_BASE_URL : DEFAULT_DEV_AURA_BASE_URL);
    } catch {
      AURA_BASE_URL = serverType === 'production' ? DEFAULT_AURA_BASE_URL : DEFAULT_DEV_AURA_BASE_URL;
    }
  } else if (serverType === 'production') {
    AURA_BASE_URL = DEFAULT_AURA_BASE_URL;
  } else {
    AURA_BASE_URL = DEFAULT_DEV_AURA_BASE_URL;
  }

  logDebug('[POPUP] API configured:', { API_BASE_URL, AURA_BASE_URL, serverType });
  return { API_BASE_URL, AURA_BASE_URL, serverType };
}

// UI Elements
const loginSection = document.getElementById('loginSection');
const dashboard = document.getElementById('dashboard');
const profileContextSection = document.getElementById('profileContextSection');

// ==================== API Helpers ====================

/**
 * Get stored token from browser storage
 */
async function getStoredToken() {
  return new Promise(resolve => {
    chrome.storage.local.get('token', result => {
      resolve(result.token || null);
    });
  });
}

/**
 * Store token in browser storage
 */
async function storeToken(token) {
  return new Promise(resolve => {
    chrome.storage.local.set({ auth_mode: 'jwt', token }, resolve);
  });
}

async function storeAuraSession(sessionId, tenantId) {
  return new Promise(resolve => {
    chrome.storage.local.set({
      auth_mode: 'aura',
      aura_session_id: sessionId,
      aura_tenant_id: tenantId || null,
      token: null
    }, resolve);
  });
}

async function getAuthState() {
  return new Promise(resolve => {
    chrome.storage.local.get(['auth_mode', 'token', 'aura_session_id', 'aura_tenant_id'], result => {
      const mode = result.auth_mode || (result.token ? 'jwt' : (result.aura_session_id ? 'aura' : null));
      resolve({
        mode,
        token: result.token || null,
        auraSessionId: result.aura_session_id || null,
        auraTenantId: result.aura_tenant_id || null
      });
    });
  });
}

function formatErrorMessage(err) {
  if (!err) return 'Unknown error';
  if (typeof err === 'string') return err;
  if (err instanceof Error) return err.message || 'Unknown error';

  const detail = err.detail ?? err.message ?? err.error;
  if (typeof detail === 'string') return detail;
  if (Array.isArray(detail)) {
    const msgs = detail
      .map(d => (typeof d === 'string' ? d : (d?.msg || d?.message || JSON.stringify(d))))
      .filter(Boolean);
    return msgs.length ? msgs.join(' | ') : 'Request failed';
  }
  if (detail && typeof detail === 'object') return JSON.stringify(detail);

  try {
    return JSON.stringify(err);
  } catch {
    return String(err);
  }
}

/**
 * Get current hostname
 */
async function getCurrentHostname() {
  return new Promise(resolve => {
    chrome.tabs.query({ active: true, currentWindow: true }, tabs => {
      if (tabs[0]) {
        try {
          const url = new URL(tabs[0].url);
          resolve(url.hostname);
        } catch {
          resolve(null);
        }
      } else {
        resolve(null);
      }
    });
  });
}

/**
 * Make authenticated API request
 */
async function apiRequest(endpoint, options = {}) {
  await loadApiConfig();
  const auth = await getAuthState();
  if (!auth.mode) throw new Error('Not authenticated');
  
  const url = `${API_BASE_URL}${endpoint}`;
  const headers = {
    'Content-Type': 'application/json',
    ...options.headers
  };

  if (auth.mode === 'jwt') {
    if (!auth.token) throw new Error('Not authenticated');
    headers['Authorization'] = `Bearer ${auth.token}`;
  } else if (auth.mode === 'aura') {
    if (!auth.auraSessionId) throw new Error('Not authenticated');
    headers['X-Session-Id'] = auth.auraSessionId;
    if (auth.auraTenantId) headers['X-Tenant-Id'] = auth.auraTenantId;
  }
  
  const response = await fetch(url, {
    ...options,
    headers
  });
  
  if (!response.ok) {
    if (response.status === 401) {
      // Token expired - logout
      await logoutUser();
      throw new Error('Session expired. Please login again.');
    }
    const error = await response.json().catch(() => ({ detail: response.statusText }));
    throw new Error(formatErrorMessage(error) || `API error: ${response.status}`);
  }
  
  return await response.json();
}

// ==================== Authentication ====================

async function login(email, password) {
  await loadApiConfig();
  const response = await fetch(`${API_BASE_URL}/auth/login`, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ email, password })
  });
  
  if (!response.ok) {
    const error = await response.json().catch(() => ({ detail: response.statusText }));
    throw new Error(formatErrorMessage(error) || 'Login failed');
  }
  
  const data = await response.json();
  await storeToken(data.access_token);
  return data;
}

async function loginAuraAdmin(username, password) {
  await loadApiConfig();
  const response = await fetch(`${AURA_BASE_URL}/api/admin/auth/login`, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({ username, password })
  });

  if (!response.ok) {
    const error = await response.json().catch(() => ({ detail: response.statusText }));
    throw new Error(formatErrorMessage(error) || 'Login failed');
  }

  const data = await response.json();
  const sessionId = data?.session_id;
  const tenantId = data?.user?.tenant_id || null;
  if (!sessionId) throw new Error('Login failed: no session ID returned');

  await storeAuraSession(sessionId, tenantId);
  return data;
}

async function register(email, password, fullName) {
  await loadApiConfig();
  const response = await fetch(`${API_BASE_URL}/auth/register`, {
    method: 'POST',
    headers: { 'Content-Type': 'application/json' },
    body: JSON.stringify({
      email,
      password,
      full_name: fullName
    })
  });
  
  if (!response.ok) {
    const error = await response.json().catch(() => ({ detail: response.statusText }));
    throw new Error(formatErrorMessage(error) || 'Registration failed');
  }
  
  return await response.json();
}

async function openRegistrationPage() {
  const { serverType } = await loadApiConfig();
  const url = serverType === 'production'
    ? 'https://altiq.io/register.html'
    : 'https://dev.altiq.io/register.html';

  chrome.tabs.create({ url });
}

function getAdminConsoleUrl(serverType) {
  return serverType === 'production'
    ? 'https://altiq.io/admin'
    : 'https://dev-admin.altiq.io/admin';
}

function getAutofillUserGuideUrl(serverType) {
  return serverType === 'production'
    ? 'https://altiq.io/admin/docs/autofill-user-guide.html'
    : 'https://dev-admin.altiq.io/admin/docs/autofill-user-guide.html';
}

async function logoutUser() {
  await chrome.storage.local.remove(['token', 'user', 'auth_mode', 'aura_session_id', 'aura_tenant_id']);
  showAuthSection();
}

// ==================== UI State Management ====================

function showAuthSection() {
  loginSection.classList.add('active');
  dashboard.classList.remove('active');
  profileContextSection.style.display = 'none';
}

async function showDashboard() {
  const auth = await getAuthState();
  if (!auth.mode) {
    showAuthSection();
    return;
  }
  
  loginSection.classList.remove('active');
  dashboard.classList.add('active');
  profileContextSection.style.display = 'block';
  
  // Load user info and stats
  await loadUserInfo();
  await loadProfileContext();
}

function showError(elementId, message) {
  const el = document.getElementById(elementId);
  if (el) {
    el.textContent = formatErrorMessage(message);
    el.classList.add('active');
    setTimeout(() => el.classList.remove('active'), 5000);
  }
}

function showSuccess(elementId, message) {
  const el = document.getElementById(elementId);
  if (el) {
    el.textContent = message;
    el.classList.add('active');
    setTimeout(() => el.classList.remove('active'), 5000);
  }
}

function hideError(elementId) {
  const el = document.getElementById(elementId);
  if (el) {
    el.classList.remove('active');
    el.textContent = '';
  }
}

// ==================== User Info & Stats ====================

async function loadUserInfo() {
  try {
    const auth = await getAuthState();
    if (auth.mode === 'aura') {
      document.getElementById('userEmail').textContent = auth.auraTenantId
        ? `Tenant: ${auth.auraTenantId}`
        : 'Admin session';
    } else {
      const user = await apiRequest('/auth/me');
      document.getElementById('userEmail').textContent = user.email;
    }
    
    // Load stats
    const context = await apiRequest('/autofill/context?hostname=');
    document.getElementById('documentCount').textContent = context.latest_doc_id ? '1+' : '0';
    document.getElementById('fieldCount').textContent = '0';
    
  } catch (error) {
    console.error('Failed to load user info:', error);
    showError('dashboardError', 'Failed to load user info');
  }
}

// ==================== Profile Context Management ====================

async function loadProfileContext() {
  try {
    const hostname = await getCurrentHostname();
    if (!hostname) return;

    logDebug('[POPUP] Loading context for hostname:', hostname);
    
    // Get context from backend
    const context = await apiRequest(`/autofill/context?hostname=${encodeURIComponent(hostname)}`);
    
    logDebug('[POPUP] Context loaded');
    
    // Update scope selector
    const scopeSelect = document.getElementById('scopeSelect');
    scopeSelect.value = context.selected_scope || 'enterprise';
    
    // Populate client dropdown for scope selection
    const scopeClientDiv = document.getElementById('scopeClientDiv');
    const scopeClientSelect = document.getElementById('scopeClientSelect');

    scopeClientSelect.replaceChildren();
    if (context.clients && context.clients.length > 0) {
      context.clients.forEach(client => {
        const option = document.createElement('option');
        option.value = client.id;
        option.textContent = client.display_name;
        scopeClientSelect.appendChild(option);
      });
      
      if (context.selected_client_id) {
        scopeClientSelect.value = context.selected_client_id;
      }
      
      // Show/hide client selector based on scope
      scopeSelect.addEventListener('change', (e) => {
        scopeClientDiv.style.display = 
          e.target.value === 'client' || e.target.value === 'both' 
            ? 'block' 
            : 'none';
      });
      
      // Trigger initial display
      if (context.selected_scope === 'client' || context.selected_scope === 'both') {
        scopeClientDiv.style.display = 'block';
      }
    }
    
    // Auto-save context changes
    scopeSelect.addEventListener('change', async (e) => {
      await saveProfileContext(hostname, e.target.value, scopeClientSelect.value);
    });
    
    scopeClientSelect.addEventListener('change', async (e) => {
      await saveProfileContext(hostname, scopeSelect.value, e.target.value);
    });
    
  } catch (error) {
    console.error('[POPUP] Failed to load context:', error);
  }
}

async function saveProfileContext(hostname, scope, clientId) {
  try {
    logDebug('[POPUP] Saving context');
    
    await apiRequest('/autofill/context', {
      method: 'PUT',
      body: JSON.stringify({
        hostname,
        selected_scope: scope,
        selected_client_id: clientId || null
      })
    });
    
    logDebug('[POPUP] Context saved successfully');
  } catch (error) {
    console.error('[POPUP] Failed to save context:', error);
  }
}

// ==================== Autofill Button ====================

async function performAutofill() {
  try {
    const auth = await getAuthState();
    if (!auth.mode) {
      showError('dashboardError', 'Not authenticated');
      return;
    }
    
    const hostname = await getCurrentHostname();
    const scope = document.getElementById('scopeSelect').value;
    const clientId = document.getElementById('scopeClientSelect').value;
    
    logDebug('[POPUP] Autofill triggered');
    
    // Send autofill command to content script. If the content script isn't present
    // (e.g., MV3 host permission constraints), inject it on-demand and retry.
    chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
      const tab = tabs?.[0];
      if (!tab?.id) {
        showError('dashboardError', 'No active tab found');
        return;
      }

      const message = { action: 'autofill', scope, clientId, hostname };

      const inject = (cb) => {
        // Ensure both CSS and JS are present before messaging.
        chrome.scripting.insertCSS({ target: { tabId: tab.id }, files: ['content.css'] }, () => {
          // Ignore CSS insert failures (some pages restrict injection).
          chrome.scripting.executeScript(
            { target: { tabId: tab.id }, files: ['content.js'] },
            () => cb()
          );
        });
      };

      const send = () => {
        chrome.tabs.sendMessage(tab.id, message, (response) => {
          if (chrome.runtime.lastError) {
            const errMsg = chrome.runtime.lastError?.message || String(chrome.runtime.lastError);
            logDebug('[POPUP] sendMessage failed:', errMsg);

            // Try injecting the content script and retry once.
            inject(() => {
                if (chrome.runtime.lastError) {
                  const injectMsg = chrome.runtime.lastError?.message || String(chrome.runtime.lastError);
                  console.error('[POPUP] content script injection failed:', injectMsg);
                  showError('dashboardError', `Could not run on this page: ${injectMsg}`);
                  return;
                }
                // Retry after successful injection
                chrome.tabs.sendMessage(tab.id, message, (response2) => {
                  if (chrome.runtime.lastError) {
                    const retryMsg = chrome.runtime.lastError?.message || String(chrome.runtime.lastError);
                    console.error('[POPUP] sendMessage retry failed:', retryMsg);
                    showError('dashboardError', 'Could not communicate with page');
                    return;
                  }
                  logDebug('[POPUP] Autofill response received');
                  showSuccess('dashboardError', 'Autofill complete!');
                });
            });
            return;
          }

          logDebug('[POPUP] Autofill response received');
          showSuccess('dashboardError', 'Autofill complete!');
        });
      };

      send();
    });
    
  } catch (error) {
    console.error('Autofill error:', error);
    showError('dashboardError', error.message || 'Autofill failed');
  }
}

// ==================== Event Listeners ====================

document.addEventListener('DOMContentLoaded', async () => {
  await loadApiConfig();
  const auth = await getAuthState();
  if (auth.mode) {
    await showDashboard();
  } else {
    showAuthSection();
  }

  // Helpful links (fill-only Phase-1)
  const { serverType } = await loadApiConfig();
  const adminUrl = getAdminConsoleUrl(serverType);
  const guideUrl = getAutofillUserGuideUrl(serverType);

  const wireLink = (id, url) => {
    const el = document.getElementById(id);
    if (!el) return;
    el.addEventListener('click', (e) => {
      e.preventDefault();
      chrome.tabs.create({ url });
    });
  };

  wireLink('adminConsoleLinkLogin', adminUrl);
  wireLink('userGuideLinkLogin', guideUrl);
  wireLink('adminConsoleLink', adminUrl);
  wireLink('userGuideLink', guideUrl);
  
  // Login form
  document.getElementById('loginBtn').addEventListener('click', async () => {
    try {
      const email = document.getElementById('loginEmail').value;
      const password = document.getElementById('loginPassword').value;
      
      if (!email || !password) {
        showError('loginError', 'Please enter username/email and password');
        return;
      }

      // If it looks like an email, use Autofill native auth; otherwise use Aura admin auth.
      if (String(email).includes('@')) {
        await login(email, password);
      } else {
        await loginAuraAdmin(email, password);
      }
      await showDashboard();
      
    } catch (error) {
      showError('loginError', error);
    }
  });
  
  // Logout
  document.getElementById('logoutBtn').addEventListener('click', async () => {
    await logoutUser();
  });
  
  // Autofill button
  document.getElementById('autofillBtn').addEventListener('click', performAutofill);
});
